/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Switch Resource Framework API Header File
::   Copyright   :   (C)2002-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_Switch.h
    \brief The framework provides a BEHAVIOUR_SWITCH behaviour for the interpretation of low frequency digital
           inputs that are most commonly used with switch type sensors. See \ref switchoverview. */

#ifndef __RESOURCE_SWITCH_H
#define __RESOURCE_SWITCH_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! Enumeration should be treated as a uint1 value that holds a percentage between \ref THRESH_MIN and
    \ref THRESH_MAX. See \ref switchthreshold. A value of \ref THRESH_DIGITAL can also be assigned, which
    indicates that the framework should select the best threshold based upon its knowledge of the hardware
    that it is executing upon.

    \code
    S_SwitchResourceAttributes ECUPInfo;
    ...
    // Set the threshold to 50% of full scale
    ECUPInfo.uValidAttributesMask = USE_SWITCH_ASSERT_THRESH;
    ECUPInfo.eAssertionThreshold = 50;
    sErrorResult = SetResourceAttributesBEHAVIOUR_SWITCH(eResource, &ECUPInfo);
    ... \endcode */
typedef enum 
{
    THRESH_MIN     = 0,     /*!< Minimum allowed threshold */
    THRESH_MAX     = 100,   /*!< Maximum allowed threshold */
    THRESH_DIGITAL = 255    /*!< Use Modules's Interpretation of a Digital Threshold for the input. */
} E_SwitchThreshold;

/*! Defines the possible software configurable pull-up settings for the switch */
/* Name: E_SwitchPullUp ClassID:ENUMDEC EnumDflt:"Illegal Pullup" */
typedef enum
{
/*! No pullup setting */
    SWITCH_PULLUP_R_NONE = 0,              /* EnumTxt:"No Pullup" */
/*! Weak pullup setting */
    SWITCH_PULLUP_R_WEAK = 1,              /* EnumTxt:"Weak Pullup" */
/*! Strong pullup setting */
    SWITCH_PULLUP_R_STRONG = 2,            /* EnumTxt:"Strong Pullup" */
} E_SwitchPullUp;

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid switch
           input attributes in the S_SwitchResourceAttributes and S_SwitchCreateResourceAttributes data structures.

Each mask definition applies to a particular member of one of the data structures.
\code
...
S_SwitchCreateResourceAttributes CreateObj;

// Will use DynamicObj and the min frequency
CreateObj.uValidAttributesMask = USE_SWITCH_DYNAMIC_ON_CREATE | USE_SWITCH_POLARITY;
CreateObj.ePolarity = RES_POL_ACTIVE_HIGH;
// Enable the switch on create and supply a threshold
CreateObj.DynamicObj.uValidAttributesMask = USE_SWITCH_THRESHOLD | USE_SWITCH_CONDITION;
CreateObj.DynamicObj.eAssertionThreshold = THRESH_DIGITAL;
CreateObj.DynamicObj.eResourceCondition = RES_ENABLED;
// The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
typedef enum
{
    USE_SWITCH_DEFAULTS             = 0x00,  /*!< Selects all defaults for the switch */
    USE_SWITCH_CONDITION            = 0x01,  /*!< Selects S_SwitchResourceAttributes::eResourceCondition */
    USE_SWITCH_ASSERT_THRESH        = 0x02,  /*!< Selects S_SwitchResourceAttributes::eAssertionThreshold */
    USE_SWITCH_DYNAMIC_ON_CREATE    = 0x04,  /*!< Selects S_SwitchCreateResourceAttributes::DynamicObj */
    USE_SWITCH_POLARITY             = 0x08,  /*!< Selects S_SwitchCreateResourceAttributes::ePolarity */
    USE_SWITCH_DEBOUNCE_SAMPLE_RATE = 0x10,  /*!< Selects S_SwitchCreateResourceAttributes::uSampleRateInMilliSecs */
    USE_SWITCH_PULLUP_STRENGTH      = 0x20,  /*!< Selects S_SwitchCreateResourceAttributes::eSwitchPullUp */

/* IF THIS TYPE EXCEEDS 0x80 THEN ALTER THE TYPE OF SwitchAttributesMask_U ACCORDINGLY */

} E_SwitchAttributesMask;

/*! Unsigned integer type of sufficient size to hold the attribute masks for an encoder described by \ref E_SwitchAttributesMask */
typedef uint1 SwitchAttributesMask_U;

/*! \brief This data structure describes all of the switch inputs resource's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_SWITCH(). The data
    structure does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit
    field member that is used to identify which attributes are valid. Each attribute is identified with a
    separate bit mask that is logic-ORed into the a mask when an attribute is to be used.
    \code
    ...
    S_SwitchResourceAttributes ECUPInfo;
    ...
    // Set the threshold to 50% of full scale
    ECUPInfo.uValidAttributesMask = USE_SWITCH_ASSERT_THRESH;
    ECUPInfo.eAssertionThreshold = 50;
    sErrorResult = SetResourceAttributesBEHAVIOUR_SWITCH(eResource, &ECUPInfo);
    ... \endcode */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_SWITCH_CONDITION and \ref USE_SWITCH_ASSERT_THRESH]
    that are valid for this this instance of the data structure.
    \code
    S_SwitchResourceAttributes ECUPInfo;
    ...
    // Set the threshold to 50% of full scale
    ECUPInfo.uValidAttributesMask = USE_SWITCH_ASSERT_THRESH;
    ECUPInfo.eAssertionThreshold = 50;
    sErrorResult = SetResourceAttributesBEHAVIOUR_SWITCH(eResource, &ECUPInfo);
    ... \endcode */
    SwitchAttributesMask_U uValidAttributesMask;
/*! The condition of the resource. A disabled resource does not execute the debounce process and will always report
    as RES_OFF regardless of what the actual state is. Use the \ref USE_SWITCH_CONDITION mask to select this attribute.
    The value of this attribute upon resource creation is \ref RES_DISABLED. */
    E_ResourceCond eResourceCondition;
/*! The assertion threshold for the switch, which is selected with the \ref USE_SWITCH_ASSERT_THRESH bit mask.
    See \ref switchthreshold */
    E_SwitchThreshold eAssertionThreshold;
} S_SwitchResourceAttributes;

typedef S_SwitchResourceAttributes const* S_SwitchResourceAttributesPtr;

/*! \brief This data structure describes the creation attributes for a switch resource

    Included within the structure is the \b DynamicObj which represents the initial states for the resource's
    dynamic attributes. This data structure is referenced when the resource is created via the
    CreateResourceBEHAVIOUR_SWITCH() function and \ref USE_SWITCH_DYNAMIC_ON_CREATE is included in the
    S_SwitchCreateResourceAttributes::uValidAttributesMask member.
    
    \code
    ...
    S_SwitchCreateResourceAttributes CreateObj;

    // Will use DynamicObj and the min frequency
    CreateObj.uValidAttributesMask = USE_SWITCH_DYNAMIC_ON_CREATE | USE_SWITCH_POLARITY;
    CreateObj.ePolarity = RES_POL_ACTIVE_HIGH;
    // Enable the switch on create and supply a threshold
    CreateObj.DynamicObj.uValidAttributesMask = USE_SWITCH_THRESHOLD | USE_SWITCH_CONDITION;
    CreateObj.DynamicObj.eAssertionThreshold = THRESH_DIGITAL;
    CreateObj.DynamicObj.eResourceCondition = RES_ENABLED;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
typedef struct
{
/*! Logic-OR the attributes [\ref USE_SWITCH_DYNAMIC_ON_CREATE, \ref USE_SWITCH_POLARITY, and \ref USE_SWITCH_DEBOUNCE_SAMPLE_RATE]
    that are valid for this this instance of the data structure.

    \code
    ...
    S_SwitchCreateResourceAttributes CreateObj;

    // Will use DynamicObj and the min frequency
    CreateObj.uValidAttributesMask = USE_SWITCH_DYNAMIC_ON_CREATE | USE_SWITCH_POLARITY;
    CreateObj.ePolarity = RES_POL_ACTIVE_HIGH;
    // Enable the switch on create and supply a threshold
    CreateObj.DynamicObj.uValidAttributesMask = USE_SWITCH_THRESHOLD | USE_SWITCH_CONDITION;
    CreateObj.DynamicObj.eAssertionThreshold = THRESH_DIGITAL;
    CreateObj.DynamicObj.eResourceCondition = RES_ENABLED;
    // The rest of thre object can remain uninitialised
    ... \endcode */
    SwitchAttributesMask_U uValidAttributesMask;
/*! Defines the polarity of the switch. Select with the \ref USE_SWITCH_POLARITY bit mask. A value of
    \ref RES_POL_ACTIVE_LOW shall be assigned if this attribute is not explicitly selected. */
    E_ResourcePolarity ePolarity;
/*! The rate at which this switch state should be sampled for the purpose of debouncing the state. Select
    this attribute with the \ref USE_SWITCH_DEBOUNCE_SAMPLE_RATE. A 50ms rate shall be assumed if this value
    is not set. Note that the rate will be rounded to suit an existing framework periodic time domain event.
    Set the value to zero to disable debounce */
    uint1 uSampleRateInMilliSecs;
/*! Initial values of the runtime attributes, which can be later altered through a call to
    SetResourceAttributesBEHAVIOUR_SWITCH(). Select with the \ref USE_SWITCH_DYNAMIC_ON_CREATE bit mask */
    S_SwitchResourceAttributes DynamicObj;

/*! Defines the strength of the software selectable pull-up on the switch input. Select this attribute
    with \ref USE_SWITCH_PULLUP_STRENGTH mask. \ref SWITCH_PULLUP_R_NONE is assumed if this attribute is 
    left undefined and it is ignored by those systems that are not able to set the strength of the pull-up */
    E_SwitchPullUp eSwitchPullUp;
} S_SwitchCreateResourceAttributes;

typedef S_SwitchCreateResourceAttributes const* S_SwitchCreateResourceAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
NativeError_S GetSwitchResourceState(E_ModuleResource,E_ResourceState* out_peState);
NativeError_S CreateResourceBEHAVIOUR_SWITCH(E_ModuleResource, S_SwitchCreateResourceAttributes const*);
NativeError_S DestroyResourceBEHAVIOUR_SWITCH(E_ModuleResource in_eResource);
NativeError_S SetResourceAttributesBEHAVIOUR_SWITCH(E_ModuleResource, S_SwitchResourceAttributes const*);

#endif /* __RESOURCE_SWITCH_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
